;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Variable declarations ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;

globals
[
  clock             ;; keeps track of the number of times modulus total-intervals through the go procedure
  num-intervals     ;; the number of time intervals a elevator can move
  total-intervals   ;; the number of clock ticks before resetting the elevators
  received-info?    ;; equals true if data has been sent from the calculators

  ;; variables for leaving a trail
  up-trail-color    ;; color of the trail left behind by an elevator as it moves upwards
  down-trail-color  ;; color of the trail left behind by an elevator as it moves downwards

  ;; plotting variables
  users             ;; list of the user ids for the current set of elevators
  elevator-to-plot  ;; the elevator to plot if plot-all-elevators? is false
  old-plot-all-elevators?  ;; the value of plot-all-elevators? in the last pass thru the go procedure

  ;; quick start instructions variables
  quick-start       ;; the current quickstart instruction displayed in the quickstart monitor
  qs-item           ;; the index of the current quickstart instruction
  qs-items          ;; the list of quickstart instructions
]

turtles-own
[
  ;; varible used in interactions with calculators
  user-id           ;; the id of a turtle which corresponds to the user-id of a calculator

  ;; position variables
  my-ycor           ;; the unwrapped vertical position of a turtle
  ycor-initial      ;; the initial vertical position of a turtle

  ;; variable to keep track of the time
  interval          ;; the current interval a turtle is moving through

  ;; variables keeping the velocities of a turtle
  velocity          ;; current velocity of an elevator
  velocities        ;; list of velocities
]

patches-own
[
  base-pcolor       ;; the original color of the patch before any trail was drawn on it
  have-trail?       ;; true if there is a trail on this patch and false otherwise
]


;;;;;;;;;;;;;;;;;;;;;
;; Setup Functions ;;
;;;;;;;;;;;;;;;;;;;;;

to startup
  setup
  setup-quick-start
  hubnet-set-client-interface "TI-83+" [ "AAA - Elevators" [ "L1" ] ]
  hubnet-reset
end

;; Initializes the display, the plots, and the global variables.
to setup
  setup-patches
  setup-vars
  setup-plots

  set-default-shape turtles "elevator"
end

;; set up the patches to have the floor numbers down the left side of the window and color all even floor a grey color
to setup-patches
  clear-output
  cp
  ct

  ;; give the floors some color to be able to distinguish one floor from another
  ask patches with [ pycor mod 2 = 0 ]
  [ set pcolor 3 ]
  ask patches with [ pycor = 0 ]
  [ set pcolor 4 ]
  ;; label each row of pycors with a floor number
  ask patches with [ pxcor = (- screen-edge-x) ]
  [ set plabel pycor ]
  ask patches
  [
    set have-trail? false
    set base-pcolor pcolor
  ]
end

;; set variables to initial values
to setup-vars
  set clock 0
  set num-intervals 9
  set total-intervals num-intervals + 3
  set received-info? false
  set up-trail-color green
  set down-trail-color red
  
  ;; by default have the elevator to plot be nobody
  set elevator-to-plot nobody
end

;; setup the position and velocity plot
to setup-plots
  locals
  [ initial-positions ]
  clear-all-plots

  ;; set the list of users to the user-id's of the existing elevators
  set users []

  if any? turtles
  [
    set users values-from turtles [ user-id ]
    set initial-positions values-from turtles [ ycor-initial ]

    set-current-plot "Elevator Position vs. Intervals"
    setup-pens false
    plot-list initial-positions true

    set-current-plot "Elevator Velocity vs. Intervals"
    setup-pens true
  ]
end

;; create pens for each of the existing elevators and color the pens to be the same color as 
;; their corresponding elevator.  if bars? is true, set the pen mode to be 1 for bar mode.
to setup-pens [ bars? ]
  locals
  [
    colors  ;; list of the colors for the current set of elevators
    index
  ]

  set index 0
  if any? turtles
  [ set colors values-from turtles [ color ] ]

  repeat length users
  [
    create-temporary-plot-pen item index users
    if bars?
    [ set-plot-pen-mode 1 ]
    set-plot-pen-color item index colors
    set index index + 1
  ]
end

;; give the user some information about what the setup button does so they can 
;; know whether they want to proceed before actually doing the setup
to setup-prompt
  if user-yes-or-no? ("The SETUP button should only be used when starting "
              + "over with a new group (such as a new set of students) since "
              + "all data is lost.  Use the RE-RUN button for continuing with "
              + "an existing group."
              + "\n\nDo you really want to setup the model?")
  [
    user-message "Before closing this dialog, please do the following:"
                 + "\n  -Have everyone that is currently logged in, log off and "
                 + "clear the calulator memory. (Press 2nd MEM 7 1 2)"
                 + "\n  -Open the teacher console for this activity and press the ERASE ALL DATA button."
    setup
  ]
end

;; creates a set of num-turtles elevators.  only used in testing.
to make-test-turtles [ num-turtles ]
  cct num-turtles
  [
    set ycor-initial 3
    set color 5 + (10 * random 14)
    set heading 0
    set interval 0
    set user-id "" + who
    ifelse (who mod 3 = 0)
    [ set velocities [ 1 2 3 2 1 0 -1 0 0 ] ]
    [
      ifelse (who mod 3 = 1)
      [ set velocities [ 0 0 1 -1 3 -2 -1 0 0 ] ]
      [ set velocities [ 0 0 -1 4 3 -2 -3 0 5 ] ]
    ]

    ;; set initial positions of the elevators  
    set my-ycor ycor-initial
    set ycor ycor-initial

    ifelse (abs my-ycor <= screen-edge-y)
    [ set hidden? false ]
    [ set hidden? true ]
    set label-color yellow
    show-or-hide-id-labels
  ]

  assign-columns

  set received-info? true
end


;;;;;;;;;;;;;;;;;;;;;;;
;; Runtime Functions ;;
;;;;;;;;;;;;;;;;;;;;;;;

;; receives information from the calculators and runs the simulation
to go
  ifelse (not received-info?)
  [
    every 1
    [
      ;; get commands and data from the calculators
      listen-calc
      if (received-info?)
      [
        re-run
        wait delay  ;; give the user some time to see the initial state of the simulation
      ]
    ]
  ]
  [
    every delay  ;; we want to move to a new interval after delay seconds
    [
      ask turtles
      [
        show-or-hide-id-labels  ;; shows or hide the turtles labels
        assign-values  ;; set the turtles' velocity to the appropriate value
        move-elevators  ;; move the elevators by their velocity for this interval
      ]
      ;; only run the plot during the number of intervals that a person can put into the calculator
      if (clock < num-intervals)
      [ do-plotting ]
      ask patches with [ have-trail? ]  ;; have patches with a trail fade back to their base color
      [ fade-trail ]
      set clock (clock + 1)
    ]
    ;; after each set of intervals, check the ti server for information
    if (clock mod total-intervals = 0)
    [
      listen-calc
      re-run
    ]
  ]
end

;; put the turtles in their initial positions and set variables to their initial values
to reset-elevators
  set clock 0
  ask patches with [ have-trail? ]
  [
    set have-trail? false
    set pcolor base-pcolor
  ]

  assign-columns
  ask turtles
  [
    set heading 0
    set ycor ycor-initial
    set interval 0
    set my-ycor ycor-initial
    ifelse (abs my-ycor <= screen-edge-y)
    [
      set ycor ycor-initial
      set hidden? false
    ]
    [ set hidden? true ]
    show-or-hide-id-labels
  ]
end

;; put the turtles in their horizontal location
to assign-columns
  ask turtles
  [ set xcor ((who + .5) * screen-size-x / (count turtles) - screen-edge-x) - .5 ]
end

;; reports the amount of seconds by which to slow the model down
to-report delay
  ifelse simulation-speed <= 0
  [ report ln (10 / 0.001) ]
  [ report ln (10 / simulation-speed) ]
end

;; based on the value of the show-user-id? switch, will either show or hide the turtles' labels
to show-or-hide-id-labels  ;; turtle procedure
  ifelse show-user-id?
  [ set label user-id + "     " ]
  [ set label no-label ]
end

;; set the turtle's velocity based on its list of velocities and what interval it is
to assign-values  ;; turtle procedure
  ifelse interval >= length velocities
  [ set velocity 0 ]
  [ set velocity item interval velocities ]
end

;; move each turtle by its velocity for this interval
to move-elevators  ;; turtle procedure
  locals
  [
    delta-pos  ;; will either be a 1 or a -1 depending upon if the velocity is positive or negative
    inner-tick  ;; an inner clock the turtles keep to make their motion smooth
    on-upper-edge?  ;; is true if the turtle is on the highest floor; otherwise it is false
    on-lower-edge?  ;; is true if the turtle is on the lowest floor; otherwise it is false
  ]

  if velocity != 0
  [ set delta-pos (velocity / abs velocity) ]
  set inner-tick 0

  while [ inner-tick < abs velocity ]
  [
    ;; divide the amount of time till the next interval into equal amounts
    ;; so as to be able to make the motion of an elevator smooth
    every ( delay / (abs velocity) )
    [
      set on-lower-edge? my-ycor = (- screen-edge-y) and my-ycor = ycor
      set on-upper-edge? my-ycor = screen-edge-y and my-ycor = ycor

      if (abs my-ycor < screen-edge-y) or on-upper-edge? or on-lower-edge?
      [
        ifelse (velocity > 0)  ;; leave a trail behind the turtle.  the color of the trail will depend upon whether the turtle is moving up or down
        [ stamp up-trail-color ]
        [ stamp down-trail-color ]
        set have-trail? true
      ]
      set my-ycor (my-ycor + delta-pos)  ;; add to the turtle's unwrapped ycor 1 or -1 based on the sign of its velocity
      ifelse (abs my-ycor <= screen-edge-y)
      [ set hidden? false ]  ;; if my-ycor is on the screen, then have the turtle color the patch that it is on
      [ set hidden? true ]  ;; otherwise, hide this turtle so it doesn't wrap around the screen, but appears to go off the edge of the screen
      fd delta-pos  ;; for each time thru the every block, move up or down by 1
      set inner-tick (inner-tick + 1)  ;; increase the inner clock of the turtle
    ]
  ]

  set interval (interval + 1)
end

;; plot the positions and velocities for the elevators in the appropriate plot
to do-plotting
  locals
  [
    vels       ;; list of velocity for each of the elevators
    positions  ;; list of the position for each elevator
  ]

  ifelse plot-all-elevators?
  [
    if (length users > 0)
    [
      set positions values-from turtles [ my-ycor ]
      set vels values-from turtles [ velocity ]
    ]
    
    set-current-plot "Elevator Position vs. Intervals"
    plot-list positions false

    set-current-plot "Elevator Velocity vs. Intervals"
    plot-list vels true
    set old-plot-all-elevators? plot-all-elevators?
  ]
  [
    ifelse( elevator-to-plot = nobody )
    [
      if plot-all-elevators? != old-plot-all-elevators?
      [
        user-message "If you want to view just 1 elevator's data in the plots, " +
                  "please use the PICK ELEVATOR TO PLOT button to pick an " +
                  "elevator to plot." +
                  "\nAlternately, switch PLOT-ALL-ELEVATORS? to true." +
                  "\nUntil you do one of these things, nothing will be plotted."
        set old-plot-all-elevators? plot-all-elevators?
      ]
    ]
    [
      set-current-plot "Elevator Position vs. Intervals"
      set-current-plot-pen value-from elevator-to-plot [ user-id ]
      plot-point (clock + 1) value-from elevator-to-plot [ my-ycor ]

      set-current-plot "Elevator Velocity vs. Intervals"
      set-current-plot-pen value-from elevator-to-plot [ user-id ]
      plot-point clock value-from elevator-to-plot [ velocity ]
    ]
  ]
  plot-x-axis "Elevator Velocity vs. Intervals"
  plot-x-axis "Elevator Position vs. Intervals"
end

;; for each element in users, plot the element in info at the same index
to plot-list 
  [
    info  ;; the list of data we are to plot
    use-clock?  ;; if true we should plot using clock as the x coordinate, otherwise, use clock + 1
  ]
  locals [ index ]

  set index 0

  repeat length users
  [
    set-current-plot-pen item index users  ;; set the pen to be the user-id at index
    ;; plot the data in info at the current index with the current pen
    ifelse use-clock?
    [ plot-point clock item index info ]
    [ plot-point (clock + 1) item index info ]
    set index index + 1
  ]
end

;; plots a black line at the x-axis of the plot this-plot
to plot-x-axis [ this-plot ]
  set-current-plot this-plot
  set-current-plot-pen "x-axis"
  plotxy plot-x-min 0
  plotxy plot-x-max 0
end

;; have any trails fade back to the base color of the patch
to fade-trail  ;; patch procedure
  set pcolor pcolor - 0.4
  if (pcolor mod 10 = 0)
  [
    set pcolor base-pcolor
    set have-trail? false
  ]
end

;; pick the column of an elevator to show in the plots
;; the plots are not shown unless plot-single? is true
to pick-elevator-to-plot
  locals
  [
    x-mouse  ;; the x coordinate of the mouse's position when it is down
    y-mouse  ;; the y coordinate of the mouse's position when it is down
  ]
  if( mouse-down? )
  [
    set x-mouse ( round mouse-xcor )
    set y-mouse ( round mouse-ycor )
    ifelse( count turtles with [ pxcor = x-mouse ] = 1 )
    [
      set elevator-to-plot one-of turtles with [ pxcor = x-mouse ]
      stop
    ]
    [
      ask patch x-mouse y-mouse
      [
        if( count turtles-here = 1 )
        [
          set elevator-to-plot one-of turtles-here
          stop
        ]
      ]
    ]
  ]
end

;; setup the model to be ready to be run with
;; the current elevators and with clock set to zero
to re-run
  reset-elevators
  setup-plots
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Quick Start functions ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; instructions to quickly setup the model, calculators, and TISchool webpage to run this activity
to setup-quick-start
  set qs-item 0
  set qs-items
  [
    "Teacher: Follow these directions to setup the HubNet activity."
    "Optional- Zoom In (see Tools in the Menu Bar) - optional"
    "Everyone: Reset the RAM on your calculator (2nd MEM 7 1 2)."
    "Login to the calculators."
    "Press ENTER to begin the simulation of the elevators."
    "Move around to acquaint yourself with the interface by:"
    "Pressing the left and right arrows to select different time intervals...."
    "Pressing the up and down arrows to adjust the velocity of..." "the elevator for the currently selected interval."
    "Press the GO softkey to play the motion described in..." "the velocity vs. time graph on the calculator."
    "Craft individual velocity vs. time graphs on the calculator."
    "Then send your graph to NetLogo by..." "pressing the SEND softkey on the calculator."
    "Teacher: Once everyone has sent the graphs,..." "press the GO button in the NetLogo model."
    "Everyone: Once a motion is sent, you can change the motion and..." "resend it by using the arrow keys and pressing..." "the SEND softkey again."
    "When the graphs are sent to the class and NetLogo,..." "collect the results via the GET softkey."
    "Cycle among the graphs on a calculator by pressing..." "ALPHA followed by LEFT or RIGHT."
    "Teacher: In NetLogo, if you wish to view just one elevator's plots,..." "press the PICK ELEVATOR TO PLOT button and click on..." "an elevator in the Graphics Window...." "Then set PLOT-ALL-ELEVATORS? to OFF."
    "To view another elevator's plots, press..." "the PICK ELEVATOR TO PLOT button again and click on..." "another elevator."
    "To view all the elevators' plots again, set..." "PLOT-ALL-ELEVATORS? to ON."
    
    "Teacher: To rerun the activity, do not clear the server."
    "Stop the simulation by pressing the NetLogo GO button."
    "Optional- Pick a single elevator to watch in the plots and..." "set PLOT-ALL-ELEVATORS? to OFF. - optional"
    "Press the RE-RUN button."
    "To restart the model, press the GO button."
    
    "Teacher: To start the simulation over with a new group,..." "stop the model by pressing the NetLogo GO button, if it is on."
    "Have everyone, including yourself, logout of their calculators."
    "Press the ERASE ALL DATA button on the TISCHOOL site."
    "Press the NetLogo SETUP button."
    "Follow these instructions from the beginning again."
  ]
  set quick-start (item qs-item qs-items)
end

;; view the next item in the quickstart monitor
to view-next
  set qs-item qs-item + 1
  if qs-item >= length qs-items
  [ set qs-item length qs-items - 1 ]
  set quick-start (item qs-item qs-items)
end

;; view the previous item in the quickstart monitor
to view-prev
  set qs-item qs-item - 1
  if qs-item < 0
  [ set qs-item 0 ]
  set quick-start (item qs-item qs-items)
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Code for interacting with the calculators ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; when a command is sent, this finds out which calculator sent it and then executes the command
to listen-calc
  locals
  [
    current-id
    cmd
  ]
  while [ hubnet-message-waiting? ]
  [
    hubnet-fetch-message
    set current-id hubnet-message-source
    set cmd hubnet-message-tag
    execute-cmd current-id cmd
    set received-info? true
  ]
end

;; Calculator command codes sent to NetLogo to indicate what each student turtle is supposed to be doing:
;; 0 - create and setup (set color, and initial position) a student turtle
to execute-cmd [ current-id cmd ]
  if cmd = "L1"
  [ setup-elevator current-id ]
end

;; Command 0 - Create a turtle, and setup its variables
;; Receives a 20 item list from the calculator.  0th item is the command. 10th item is initial position
;; Items 1-10 are obsolete.  Items 11-19 hold the velocities of the turtle.
to setup-elevator [ current-id ]
  ifelse not any? turtles with [user-id = current-id]
  [
    cct 1
    [
      set user-id current-id
      set velocities [0 0 0 0 0 0 0 0 0]
      set heading 0
      set interval 0
      set color 5 + 10 * random 14
      set ycor-initial item 10 hubnet-message
      set ycor ycor-initial
      setup-velocities
      ifelse (abs my-ycor <= screen-edge-y)
      [ set hidden? false ]
      [ set hidden? true ]
      set label-color yellow
      show-or-hide-id-labels
    ]
  ]
  [
    ask turtles with [ user-id = current-id ]
    [
      set heading 0
      set interval 0
      set ycor-initial item 10 hubnet-message
      setup-velocities
    ]
  ]
end

;; copy the velocities from the calculator to the velocities list
to setup-velocities  ;; turtle procedure
  locals
  [
    index  ;; temporary index variable to place info into the velocities lists
  ]

  set index 0
  repeat length velocities
  [
    set velocities replace-item index velocities item (index + 11) hubnet-message
    set index index + 1
  ]
end


; ***NetLogo Model Copyright Notice***

; This activity and associated models and materials was created as part of the project:
; PARTICIPATORY SIMULATIONS: NETWORK-BASED DESIGN FOR SYSTEMS LEARNING IN CLASSROOMS.
; The project gratefully acknowledges the support of the 
; National Science Foundation ( REPP program ) -- grant number REC #9814682.

; Copyright 2000 by Uri Wilensky & Walter Stroup. Updated 2001,2002. All rights reserved.

; Permission to use, modify or redistribute this model is hereby granted,
; provided that both of the following requirements are followed:
; a) this copyright notice is included.
; b) this model will not be redistributed for profit without permission
;    from the copyright holders.
; Contact the copyright holders for appropriate licenses for redistribution for
; profit.

; To refer to this model in academic publications, please use:
; Wilensky, U. & Stroup, W. (2000).  NetLogo HubNet Elevators model.
; http://ccl.northwestern.edu/netlogo/models/HubNetElevators.
; Center for Connected Learning and Computer-Based Modeling,
; Northwestern University, Evanston, IL.

; In other publications, please use:
; Copyright 1998 by Uri Wilensky and Walter Stroup.  All rights reserved.  See
; http://ccl.northwestern.edu/netlogo/models/HubNetElevators
; for terms of use.
;
; ***End NetLogo Model Copyright Notice***
@#$#@#$#@
GRAPHICS-WINDOW
187
98
752
414
18
9
15.0
1
12
1
1
1
0

CC-WINDOW
5
664
761
759
Command Center

BUTTON
79
61
146
94
Go 
go
T
1
T
OBSERVER
T
NIL

BUTTON
5
10
73
43
Setup
setup-prompt
NIL
1
T
OBSERVER
T
NIL

MONITOR
8
176
179
225
Current Time Interval
clock
0
1

BUTTON
152
61
243
94
Go Once
go
NIL
1
T
OBSERVER
T
NIL

MONITOR
256
10
655
59
Quick Start Instructions (more details in info tab)
quick-start
0
1

BUTTON
476
60
554
93
<<< PREV
view-prev
NIL
1
T
OBSERVER
T
NIL

BUTTON
553
60
637
93
NEXT >>>
view-next
NIL
1
T
OBSERVER
T
NIL

BUTTON
274
60
392
93
Reset Instructions
setup-quick-start
NIL
1
T
OBSERVER
T
NIL

SLIDER
8
138
179
171
simulation-speed
simulation-speed
0
10.0
2.5
0.1
1
NIL

PLOT
5
421
279
650
Elevator Position vs. Intervals
Intervals
Position
0.0
9.0
-9.0
9.0
false
false
PENS
"x-axis" 1.0 0 -16777216 true

PLOT
280
421
553
649
Elevator Velocity vs. Intervals
Intervals
Velocity
0.0
9.0
-6.0
6.0
false
false
PENS
"x-axis" 1.0 0 -16777216 true

BUTTON
9
234
178
267
Pick Elevator To Plot
pick-elevator-to-plot
T
1
T
OBSERVER
T
NIL

SWITCH
9
272
178
305
plot-all-elevators?
plot-all-elevators?
0
1
-1000

BUTTON
114
10
205
43
Re-Run
re-run
NIL
1
T
OBSERVER
T
NIL

SWITCH
9
330
178
363
show-user-id?
show-user-id?
1
1
-1000

@#$#@#$#@
WHAT IS IT?
-----------
In the calculator program, each student defines the motion of an elevator by setting its velocity as a function of time.  The students have 9 different intervals for which they can set the velocity of the elevator.  They can then send these velocities to NetLogo, where the actions of each elevator can be displayed next to the others.  This can serve as a jumping off point for concepts ranging from derivatives and integrals to wave mechanics.

This activity requires the TI Navigator activity AAA - Elevators

For further documentation, see the Participatory Simulations Guide found at http://ccl.northwestern.edu/ps/


HOW TO USE IT
-------------

QUICKSTART INSTRUCTIONS:
------------------------
Contains instructions as to how to quickly setup the model, calculators, and TISchool web page so as to run this activity.  The instructions can be found below:

Teacher: Open the TISchool site in your web browser.
Enable activity AAA - Elevators.
Open the teacher console and press the ERASE ALL DATA button.

If it is not open already, open the NetLogo model.
If you are prompted by a Login dialog, enter your teacher id and password and any other necessary information.
Optional- Zoom In (see Tools in the Menu Bar)

Everyone: Reset the RAM on the calculator by pressing the following keys: 2nd MEM 7 1 2.
Login to the calculators.
Press ENTER to begin the simulation of the elevators.
Move around to acquaint yourself with the interface.  Pressing the left and right arrows to select different time intervals.  Pressing the up and down arrows to adjust the velocity of the elevator for the currently selected interval.
Press the GO softkey to play the motion described in the velocity vs. time graph on the calculator.

Craft individual velocity vs. time graphs on the calculator.
Then send your graph to NetLogo by pressing the SEND softkey on the calculator.
Teacher: Once everyone has sent the graphs, press the GO button in the NetLogo model.
Once a motion is sent, you can change the motion and resend it by using the arrow keys and pressing the SEND softkey again.

When the graphs are sent to the class and NetLogo, collect the results via the GET softkey.  Cycle among the graphs on a calculator by pressing ALPHA followed by LEFT or RIGHT.

In NetLogo, if you wish to view just one elevator's plots, press the PICK ELEVATOR TO PLOT button and click on an elevator in the Graphics Window.  Then set PLOT-ALL-ELEVATORS? to OFF.  To view another elevator's plots, press the PICK ELEVATOR TO PLOT button again and click on another elevator.  To view all the elevators' plots again, set PLOT-ALL-ELEVATORS? to ON.

Teacher: To rerun the activity, do not clear the server.
Stop the simulation by pressing the NetLogo GO button.
Optional- Pick a single elevator to watch in the plots and set PLOT-ALL-ELEVATORS? to OFF.
Press the RE-RUN button.
To restart the model, press the GO button.

Teacher: To start the simulation over with a new group, stop the model by pressing the NetLogo GO button, if it is on.
Have everyone, including yourself, logout of their calculators.
Press the ERASE ALL DATA button on the TISCHOOL site.
Press the NetLogo SETUP button.
Follow these instructions from the beginning again.

BUTTONS:
--------
SETUP - clears all turtles, and draws lines marking the floors
RE-RUN - sets up the model to be able to run with the current elevators and with clock set to zero
GO ONCE - moves the turtles by 1 interval
GO - receives information from the calculators and runs the simulation
PICK ELEVATOR TO PLOT - allows the user to pick an elevator to plot the velocity and position information for.
NEXT >>> - shows the next quick start instruction
<<< PREVIOUS - shows the previous quick start instruction
RESET INSTRUCTIONS - shows the first quick start instruction

SLIDERS:
--------
SIMULATION-SPEED - the speed the simulation runs at.  The higher the number the faster it runs.

SWITCHES:
---------
PLOT-ALL-ELEVATORS? - if true, the data for all the elevators will be shown in the plots.  If it is false and if an elevator has been picked by using the PICK ELEVATOR TO PLOT button, only the picked elevator's information will be shown in the plot from the current time tick forward.  Old plot data is not erased.
SHOW-USER-ID? - if true, all the turtles show their user-id as its label.  If false, no label is shown.


CALCULATOR INFORMATION
----------------------

TEACHER CALCULATOR:
-------------------
Students and teacher have identical calculator programs.

STUDENT CALCULATOR:
-------------------
To change the initial position of the elevator press 0 or LEFT / RIGHT until 0 is indicated as the selected interval in the upper left corner of the screen.
To change the velocity during a time interval, press the number of that interval (1 - 9) or LEFT / RIGHT until that interval is indicated in the upper left of the screen.
To view the motion of the elevator select the GO softkey or press ALPHA G
To toggle between the velocity time and position time graphs press the DISP softkey or ALPHA D

To clear the graph, press PRGM or press the CLR softkey.

If you have sent your graph, and then cleared it, you may recall it by pressing the RCLL softkey.

To quit, press 2nd QUIT or press the QUIT softkey.

To send the motion of your calculator to the class and to NetLogo, press the SEND softkey or ALPHA S.  When the graphs are sent to the class and NetLogo, everyone can collect the results via the GET softkey or ALPHA R.  To cycle among the graphs on a calculator, press ALPHA followed by LEFT or RIGHT.


THINGS TO NOTICE
----------------
Notice if an elevator moves more than 1 floor during an interval, it equally divides the amount of time spent traveling through each floor.


THINGS TO TRY
-------------
Identify your elevator (necessary even with the first use!)
Have your elevator sit at the third floor during the fourth time segment.  What is the same / different between the solutions?
Make a traveling / standing wave.
Have a velocity of +2 during the fourth time interval.
Start and End at the same floor.


EXTENDING THE MODEL
-------------------
A real elevator does not move at a constant speed when moving from floor to floor.  It will build its speed gradually until it has reached a constant speed.  Then before reaching the floor it is supposed to stop on, it will slow down.  Try to make this happen as a part of the model.


CREDITS AND REFERENCES
----------------------
This activity and associated models and materials was created as part of the project:
PARTICIPATORY SIMULATIONS: NETWORK-BASED DESIGN FOR SYSTEMS LEARNING IN CLASSROOMS.
The project gratefully acknowledges the support of the National Science Foundation ( REPP program ) -- grant number REC #9814682.

Copyright 2000 by Uri Wilensky & Walter Stroup. Updated 2001,2002. All rights reserved.

Permission to use, copy, or modify this software and its associated documentation, models and curricular materials for educational and research purposes only and without fee is hereby granted, provided that this copyright notice and the original authors' names appear on all copies and supporting documentation. For any other uses of this software, in original or modified form, including, but not limited to distribution in whole or in part, specific prior permission must be obtained from Uri Wilensky and Walter Stroup. These programs shall not be used, rewritten, or adapted as the basis of a commercial or hardware product without first obtaining appropriate licenses from Uri Wilensky & Walter Stroup. We make no representations about the suitability of this software for any purpose. It is provided "as is" without express or implied warranty.

To refer to this model in academic publications, please use: Wilensky, U. & Stroup, W. (2000).  NetLogo HubNet Elevators model. http://ccl.northwestern.edu/netlogo/models/HubNetElevators. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

In other publications, please use: Copyright 2000 by Uri Wilensky and Walter Stroup.  All rights reserved.  See http://ccl.northwestern.edu/netlogo/models/HubNetElevators for terms of use.
@#$#@#$#@
default
true
0
Polygon -7566196 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7566196 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7566196 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7566196 true true 150 285 285 225 285 75 150 135
Polygon -7566196 true true 150 135 15 75 150 15 285 75
Polygon -7566196 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7566196 true true 96 182 108
Circle -7566196 true true 110 127 80
Circle -7566196 true true 110 75 80
Line -7566196 true 150 100 80 30
Line -7566196 true 150 100 220 30

butterfly
true
0
Polygon -7566196 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7566196 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7566196 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7566196 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7566196 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7566196 true true 47 195 58
Circle -7566196 true true 195 195 58

circle
false
0
Circle -7566196 true true 30 30 240

circle 2
false
0
Circle -7566196 true true 16 16 270
Circle -16777216 true false 46 46 210

cow
false
0
Polygon -7566196 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7566196 true true 73 210 86 251 62 249 48 208
Polygon -7566196 true true 25 114 16 195 9 204 23 213 25 200 39 123

face happy
false
0
Circle -7566196 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7566196 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7566196 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7566196 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7566196 true true 60 15 75 300
Polygon -7566196 true true 90 150 270 90 90 30
Line -7566196 true 75 135 90 135
Line -7566196 true 75 45 90 45

flower
false
0
Polygon -11352576 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7566196 true true 85 132 38
Circle -7566196 true true 130 147 38
Circle -7566196 true true 192 85 38
Circle -7566196 true true 85 40 38
Circle -7566196 true true 177 40 38
Circle -7566196 true true 177 132 38
Circle -7566196 true true 70 85 38
Circle -7566196 true true 130 25 38
Circle -7566196 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -11352576 true false 189 233 219 188 249 173 279 188 234 218
Polygon -11352576 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7566196 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7566196 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7566196 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7566196 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7566196 true 150 0 150 300

pentagon
false
0
Polygon -7566196 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7566196 true true 110 5 80
Polygon -7566196 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7566196 true true 127 79 172 94
Polygon -7566196 true true 195 90 240 150 225 180 165 105
Polygon -7566196 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7566196 true true 135 90 165 300
Polygon -7566196 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7566196 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7566196 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7566196 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7566196 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7566196 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7566196 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7566196 true true 30 30 270 270

square 2
false
0
Rectangle -7566196 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7566196 true true 60 270 150 0 240 270 15 105 285 105
Polygon -7566196 true true 75 120 105 210 195 210 225 120 150 75

target
false
0
Circle -7566196 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7566196 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7566196 true true 120 120 60

tree
false
0
Circle -7566196 true true 118 3 94
Rectangle -6524078 true false 120 195 180 300
Circle -7566196 true true 65 21 108
Circle -7566196 true true 116 41 127
Circle -7566196 true true 45 90 120
Circle -7566196 true true 104 74 152

triangle
false
0
Polygon -7566196 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7566196 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7566196 true true 4 45 195 187
Polygon -7566196 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7566196 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7566196 false true 24 174 42
Circle -7566196 false true 144 174 42
Circle -7566196 false true 234 174 42

turtle
true
0
Polygon -11352576 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -11352576 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -11352576 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -11352576 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -11352576 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7566196 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7566196 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7566196 true 150 285 150 15
Line -7566196 true 15 150 285 150
Circle -7566196 true true 120 120 60
Line -7566196 true 216 40 79 269
Line -7566196 true 40 84 269 221
Line -7566196 true 40 216 269 79
Line -7566196 true 84 40 221 269

x
false
0
Polygon -7566196 true true 270 75 225 30 30 225 75 270
Polygon -7566196 true true 30 75 75 30 270 225 225 270

elevator
false
0
Rectangle -1 true false 45 14 255 286
Circle -7566196 true true 115 16 69
Rectangle -7566196 true true 139 82 159 101
Rectangle -7566196 true true 109 92 192 206
Polygon -7566196 true true 191 109 248 127 234 164 190 128
Polygon -7566196 true true 109 109 53 131 62 162 109 126
Rectangle -7566196 true true 118 205 146 278
Rectangle -7566196 true true 157 205 181 277
Rectangle -7566196 true true 180 264 203 277
Rectangle -7566196 true true 92 265 118 278

@#$#@#$#@
NetLogo 2.1beta3
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
